<?php

namespace App\Services;

use App\Models\CompanyInformation;
use App\Models\CompanyContentBlock;
use App\Models\CompanyContentBlockImage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CompanyService extends BaseService
{
    protected $model;
    protected $contentBlockModel;
    protected $contentBlockImageModel;

    public function __construct(
        CompanyInformation $model,
        CompanyContentBlock $contentBlockModel,
        CompanyContentBlockImage $contentBlockImageModel
    ) {
        $this->model = $model;
        $this->contentBlockModel = $contentBlockModel;
        $this->contentBlockImageModel = $contentBlockImageModel;
    }

    public function getCompanyInfo()
    {
        return $this->model
            ->with(['contentBlocks' => function($query) {
                $query->orderBy('display_order', 'asc')
                      ->with(['images' => function($query) {
                          $query->orderBy('display_order', 'asc');
                      }]);
            }])
            ->first();
    }

    public function updateCompanyInfo(array $data)
    {
        DB::beginTransaction();
        try {
            $company = $this->model->first();
            
            if (!$company) {
                $company = $this->model->create([
                    'title' => $data['title'],
                    'subtitle' => $data['subtitle'],
                    'number_of_blocks' => count($data['blocks'])
                ]);
            } else {
                $company->update([
                    'title' => $data['title'],
                    'subtitle' => $data['subtitle'],
                    'number_of_blocks' => count($data['blocks'])
                ]);
            }

            // Store existing images information before deleting blocks
            $existingImages = [];
            foreach ($company->contentBlocks as $block) {
                foreach ($block->images as $image) {
                    $existingImages[$image->image_path] = $image->image_path;
                }
            }

            // First delete content block images to avoid foreign key constraint violations
            foreach ($company->contentBlocks as $block) {
                $this->contentBlockImageModel->where('content_block_id', $block->id)->delete();
            }
            
            // Now delete content blocks
            $this->contentBlockModel->where('company_information_id', $company->id)->delete();

            // Track which images are being reused
            $reusedImages = [];

            // Create new content blocks
            if (isset($data['blocks'])) {
                foreach ($data['blocks'] as $blockData) {
                    $contentBlock = $this->contentBlockModel->create([
                        'company_information_id' => $company->id,
                        'content' => $blockData['content'],
                        'display_order' => (int)$blockData['display_order'],
                        'image_position' => $blockData['image_position']
                    ]);

                    $displayOrder = 1;

                    // Handle existing images
                    if (isset($blockData['existing_images']) && is_array($blockData['existing_images'])) {
                        foreach ($blockData['existing_images'] as $imagePath) {
                            // Extract the relative path from the full URL
                            $relativePath = str_replace('http://127.0.0.1:8000/', '', $imagePath);
                            
                            // If the image exists in our tracked existing images
                            if (isset($existingImages[$relativePath])) {
                                $this->contentBlockImageModel->create([
                                    'content_block_id' => $contentBlock->id,
                                    'image_path' => $relativePath,
                                    'display_order' => $displayOrder++
                                ]);
                                $reusedImages[$relativePath] = true;
                            }
                        }
                    }

                    // Handle new uploaded images
                    if (isset($blockData['images']) && is_array($blockData['images'])) {
                        foreach ($blockData['images'] as $image) {
                            if ($image instanceof \Illuminate\Http\UploadedFile && $image->isValid()) {
                                $imagePath = $this->uploadFile($image, 'company/content-blocks');
                                
                                if ($imagePath) {
                                    $this->contentBlockImageModel->create([
                                        'content_block_id' => $contentBlock->id,
                                        'image_path' => $imagePath,
                                        'display_order' => $displayOrder++
                                    ]);
                                }
                            }
                        }
                    }
                }
            }

            // Delete images that are no longer used
            foreach ($existingImages as $imagePath) {
                if (!isset($reusedImages[$imagePath])) {
                    if (file_exists(public_path($imagePath))) {
                        $this->deleteFile($imagePath);
                    }
                }
            }

            DB::commit();
            return $this->getCompanyInfo();
        } catch (\Exception $e) {
            DB::rollback();
            throw $e;
        }
    }

    public function addCompanyContentBlockImage(int $contentBlockId, array $imageData)
    {
        return $this->contentBlockImageModel->create([
            'content_block_id' => $contentBlockId,
            'image_path' => $imageData['image_path'],
            'display_order' => $imageData['display_order'] ?? 0,
        ]);
    }

    public function updateContentBlocks(int $companyId, array $contentBlocks)
    {
        // Clear existing content blocks
        $this->contentBlockModel->where('company_id', $companyId)->delete();
        
        // Add new content blocks
        foreach ($contentBlocks as $index => $block) {
            $this->contentBlockModel->create([
                'company_id' => $companyId,
                'heading' => $block['heading'] ?? null,
                'content' => $block['content'],
                'display_order' => $index
            ]);
        }
    }
}