<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\PortfolioProject;
use App\Http\Resources\PortfolioProjectResource;
use App\Http\Requests\Portfolio\StoreProjectRequest;
use App\Http\Requests\Portfolio\UpdateProjectRequest;
use App\Services\PortfolioService;
use App\Traits\ApiResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class PortfolioController extends Controller
{
    use ApiResponse;

    protected $portfolioService;

    public function __construct(PortfolioService $portfolioService)
    {
        $this->portfolioService = $portfolioService;
    }


    public function index(Request $request)
    {
        try {

            $query = PortfolioProject::with(['category', 'images', 'features', 'whyChooseUs', 'testimonials'])->active()->ordered();
            
            if ($request->has('category_id')) {
                $query->byCategory($request->category_id);
            }
            
            $projects = $query->get();
            return $this->successResponse(
                PortfolioProjectResource::collection($projects)
            );

        } catch (\Exception $e) {
            return $this->errorResponse('Error fetching projects: ' . $e->getMessage());
        }
    }


    public function store(StoreProjectRequest $request)
    {
        try {
            $project = $this->portfolioService->createProject(
                $request->except(['images', 'captions']),
                $request->file('thumbnail'),
                $request->file('images', []),
                $request->input('captions', [])
            );

            return $this->createdResponse(
                new PortfolioProjectResource($project->load(['category', 'images']))
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error creating project: ' . $e->getMessage(), 500);
        }
    }


    public function show(PortfolioProject $portfolioProject)
    {
        try {
            return $this->successResponse(
                new PortfolioProjectResource($portfolioProject->load(['category', 'images', 'features']))
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error fetching project: ' . $e->getMessage());
        }
    }

    public function showBySlug($slug)
    {
        try {
            $project = PortfolioProject::where('slug', $slug)
                ->with(['category', 'images', 'features', 'whyChooseUs', 'testimonials'])
                ->first();

            if (!$project) {
                return $this->errorResponse('Project not found.', 404);
            }

            return $this->successResponse(new PortfolioProjectResource($project));

        } catch (\Exception $e) {
            return $this->errorResponse('Error fetching project: ' . $e->getMessage());
        }
    }


    public function update(UpdateProjectRequest $request, PortfolioProject $portfolioProject)
    {
        try {
            $updatedProject = $this->portfolioService->updateProject(
                $portfolioProject,
                $request->except(['new_images', 'new_captions', 'remove_images']),
                $request->file('thumbnail'),
                $request->file('new_images', []),
                $request->input('new_captions', []),
                $request->input('remove_images', [])
            );

            return $this->successResponse(
                new PortfolioProjectResource($updatedProject->load(['category', 'images', 'features', 'whyChooseUs', 'testimonials']))
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error updating project: ' . $e->getMessage(), 500);
        }
    }


    public function destroy(PortfolioProject $portfolioProject)
    {
        try {
            $this->portfolioService->deleteProject($portfolioProject);
            return $this->deletedResponse();
        } catch (\Exception $e) {
            return $this->errorResponse('Error deleting project: ' . $e->getMessage(), 500);
        }
    }
}
