<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\PortfolioCategory;
use App\Http\Resources\PortfolioCategoryResource;
use App\Http\Requests\Portfolio\StoreCategoryRequest;
use App\Http\Requests\Portfolio\UpdateCategoryRequest;
use App\Services\PortfolioCategoryService;
use App\Traits\ApiResponse;

class PortfolioCategoryController extends Controller
{
    use ApiResponse;

    protected PortfolioCategoryService $categoryService;

    public function __construct(PortfolioCategoryService $categoryService)
    {
        $this->categoryService = $categoryService;
    }

    public function index()
    {
        try {
            $categories = $this->categoryService->getAllWithProjectCounts();
            return $this->successResponse(
                PortfolioCategoryResource::collection($categories)
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error fetching categories: ' . $e->getMessage());
        }
    }

    public function store(StoreCategoryRequest $request)
    {
        try {
            $category = $this->categoryService->create($request->validated());
            return $this->createdResponse(
                new PortfolioCategoryResource($category)
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error creating category: ' . $e->getMessage(), 500);
        }
    }

    public function show(PortfolioCategory $portfolioCategory)
    {
        try {
            return $this->successResponse(
                new PortfolioCategoryResource($portfolioCategory->load('projects'))
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error fetching category: ' . $e->getMessage());
        }
    }

    public function update(UpdateCategoryRequest $request, PortfolioCategory $portfolioCategory)
    {
        try {
            $category = $this->categoryService->update($portfolioCategory, $request->validated());
            return $this->successResponse(
                new PortfolioCategoryResource($category)
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error updating category: ' . $e->getMessage(), 500);
        }
    }

    public function destroy(PortfolioCategory $portfolioCategory)
    {
        try {
            $this->categoryService->delete($portfolioCategory);
            return $this->deletedResponse();
        } catch (\Exception $e) {
            return $this->errorResponse('Error deleting category: ' . $e->getMessage(), 500);
        }
    }
}