<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Article;
use App\Http\Resources\ArticleResource;
use App\Http\Requests\Article\StoreArticleRequest;
use App\Http\Requests\Article\UpdateArticleRequest;
use App\Services\ArticleService;
use App\Traits\ApiResponse;
use Illuminate\Http\Request;

class ArticleController extends Controller
{
    use ApiResponse;

    protected ArticleService $articleService;

    public function __construct(ArticleService $articleService)
    {
        $this->articleService = $articleService;
    }

    public function index(Request $request)
    {
        try {
            $articles = $this->articleService->getPaginatedArticles($request);
            return $this->successResponse(
                ArticleResource::collection($articles)
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error fetching articles: ' . $e->getMessage());
        }
    }

    public function store(StoreArticleRequest $request)
    {
        try {
            $article = $this->articleService->create($request->validated());
            return $this->createdResponse(
                new ArticleResource($article->load('category'))
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error creating article: ' . $e->getMessage(), 500);
        }
    }

    public function show(Article $article)
    {
        try {
            return $this->successResponse(
                new ArticleResource($article->load('category'))
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error fetching article: ' . $e->getMessage());
        }
    }

    public function update(UpdateArticleRequest $request, Article $article)
    {
        try {
            $article = $this->articleService->update($article, $request->validated());
            return $this->successResponse(
                new ArticleResource($article->load('category'))
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error updating article: ' . $e->getMessage(), 500);
        }
    }

    public function destroy(Article $article)
    {
        try {
            $this->articleService->delete($article);
            return $this->deletedResponse();
        } catch (\Exception $e) {
            return $this->errorResponse('Error deleting article: ' . $e->getMessage(), 500);
        }
    }

    public function bySlug($slug)
    {
        try {
            $article = $this->articleService->findBySlug($slug);
            return $this->successResponse(
                new ArticleResource($article->load('category'))
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error fetching article: ' . $e->getMessage());
        }
    }
}
