<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Testimonial;
use App\Http\Resources\TestimonialResource;
use App\Http\Requests\Testimonial\StoreTestimonialRequest;
use App\Http\Requests\Testimonial\UpdateTestimonialRequest;
use App\Services\TestimonialService;
use App\Traits\ApiResponse;
use Illuminate\Support\Facades\Log;

class TestimonialController extends Controller
{
    use ApiResponse;

    protected TestimonialService $testimonialService;

    public function __construct(TestimonialService $testimonialService)
    {
        $this->testimonialService = $testimonialService;
    }

    public function index()
    {
        try {
            $testimonials = $this->testimonialService->getAllActive();
            return $this->successResponse(
                TestimonialResource::collection($testimonials)
            );
        } catch (\Exception $e) {
            return $this->errorResponse(
                'Error fetching testimonials: ' . $e->getMessage()
            );
        }
    }

    public function store(StoreTestimonialRequest $request)
    {
        try {
            $testimonial = $this->testimonialService->create($request->validated());
            return $this->createdResponse(
                new TestimonialResource($testimonial)
            );
        } catch (\Exception $e) {
            return $this->errorResponse(
                'Error creating testimonial: ' . $e->getMessage(), 500
            );
        }
    }

    public function show(Testimonial $testimonial)
    {
        try {
            return $this->successResponse(
                new TestimonialResource($testimonial)
            );
        } catch (\Exception $e) {
            return $this->errorResponse(
                'Error fetching testimonial: ' . $e->getMessage()
            );
        }
    }

    public function update(UpdateTestimonialRequest $request, Testimonial $testimonial)
    {
        try {
            $testimonial = $this->testimonialService->update($testimonial, $request->validated());
            return $this->successResponse(
                new TestimonialResource($testimonial)
            );
        } catch (\Exception $e) {
            return $this->errorResponse(
                'Error updating testimonial: ' . $e->getMessage(), 500
            );
        }
    }

    public function destroy(Testimonial $testimonial)
    {
        try {
            $this->testimonialService->delete($testimonial);
            return $this->deletedResponse();
        } catch (\Exception $e) {
            return $this->errorResponse(
                'Error deleting testimonial: ' . $e->getMessage(), 500
            );
        }
    }
}
