<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Service;
use App\Http\Resources\ServiceResource;
use App\Http\Requests\Service\StoreServiceRequest;
use App\Http\Requests\Service\UpdateServiceRequest;
use App\Services\ServiceService;
use App\Traits\ApiResponse;

class ServiceController extends Controller
{
    use ApiResponse;

    protected ServiceService $serviceService;

    public function __construct(ServiceService $serviceService)
    {
        $this->serviceService = $serviceService;
    }

    public function index()
    {
        try {
            $services = $this->serviceService->getAllActive();
            return $this->successResponse(
                ServiceResource::collection($services)
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error fetching services: ' . $e->getMessage());
        }
    }

    public function store(StoreServiceRequest $request)
    {
        try {
            $service = $this->serviceService->create($request->validated());
            return $this->createdResponse(
                new ServiceResource($service)
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error creating service: ' . $e->getMessage(), 500);
        }
    }

    public function show(Service $service)
    {
        try {
            return $this->successResponse(
                new ServiceResource($service)
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error fetching service: ' . $e->getMessage());
        }
    }

    public function update(UpdateServiceRequest $request, Service $service)
    {
        try {
            $service = $this->serviceService->update($service, $request->validated());
            return $this->successResponse(
                new ServiceResource($service)
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error updating service: ' . $e->getMessage(), 500);
        }
    }

    public function destroy(Service $service)
    {
        try {
            $this->serviceService->delete($service);
            return $this->deletedResponse();
        } catch (\Exception $e) {
            return $this->errorResponse('Error deleting service: ' . $e->getMessage(), 500);
        }
    }
}


