<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\ArticleCategory;
use App\Http\Resources\ArticleCategoryResource;
use App\Http\Requests\Article\StoreCategoryRequest;
use App\Http\Requests\Article\UpdateCategoryRequest;
use App\Services\ArticleCategoryService;
use App\Traits\ApiResponse;

class ArticleCategoryController extends Controller
{
    use ApiResponse;

    protected ArticleCategoryService $categoryService;

    public function __construct(ArticleCategoryService $categoryService)
    {
        $this->categoryService = $categoryService;
    }

    public function index()
    {
        try {
            $categories = $this->categoryService->getAllWithArticleCounts();
            return $this->successResponse(
                ArticleCategoryResource::collection($categories)
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error fetching categories: ' . $e->getMessage());
        }
    }

    public function store(StoreCategoryRequest $request)
    {
        try {
            $category = $this->categoryService->create($request->validated());
            return $this->createdResponse(
                new ArticleCategoryResource($category)
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error creating category: ' . $e->getMessage(), 500);
        }
    }

    public function show(ArticleCategory $articleCategory)
    {
        try {
            return $this->successResponse(
                new ArticleCategoryResource($articleCategory->load('articles'))
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error fetching category: ' . $e->getMessage());
        }
    }

    public function update(UpdateCategoryRequest $request, ArticleCategory $articleCategory)
    {
        try {
            $category = $this->categoryService->update($articleCategory, $request->validated());
            return $this->successResponse(
                new ArticleCategoryResource($category)
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error updating category: ' . $e->getMessage(), 500);
        }
    }

    public function destroy(ArticleCategory $articleCategory)
    {
        try {
            $this->categoryService->delete($articleCategory);
            return $this->deletedResponse();
        } catch (\Exception $e) {
            return $this->errorResponse('Error deleting category: ' . $e->getMessage(), 500);
        }
    }
} 